/*
 * @brief SCTimer/PWM (SCT) Match Reload example
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "board.h"


/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
#define DC1        (130)                                   				// duty cycle 1
#define DC2        (135)                                   				// duty cycle 2
#define hperiod    (180)

/*****************************************************************************
 * Private functions
 ****************************************************************************/

/*****************************************************************************
 * Public functions
 ****************************************************************************/

void SCT_Init(void)
{

	Chip_SCT_Init(LPC_SCT);			                  		 														// enable the SCT clock

	Chip_SCT_Config(LPC_SCT, SCT_CONFIG_16BIT_COUNTER  |													// split timers, auto limit
                            SCT_CONFIG_AUTOLIMIT_L    );

	Chip_Clock_SetMain_A_ClockSource(SYSCON_MAIN_A_CLKSRC_FRO12MHZ);							// Set Main Clock (SCT Clock) as 12MHz

	Chip_SCT_SetControl(LPC_SCT, SCT_CTRL_BIDIR_L(1));     												// configure SCT as BIDIR


	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_0, hperiod);												//match on (half) PWM period

	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_1, DC1);														// match on duty cycle 1

	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_2, DC2);														// match on duty cycle 2

	Chip_SCT_EventStateMask(LPC_SCT, 0, ENABLE_ALL_STATES);	 											// event 0 happens in all states
	Chip_SCT_EventControl(LPC_SCT, 0, ( SCT_EVECTRL_MATCH1  |
                                      SCT_COMBMODE_MATCH  ));										// match 1 (DC1) only condition

	Chip_SCT_EventStateMask(LPC_SCT, 1, ENABLE_ALL_STATES);	 											// event 1 happens in all states
	Chip_SCT_EventControl(LPC_SCT, 1,  ( SCT_EVECTRL_MATCH2 |
                                       SCT_COMBMODE_MATCH  )); 									// match 2 (DC) only condition

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_4, SCT_EVT_0);													// event 0 set OUT4 (green LED)
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_4, SCT_EVT_0);												//  event 0 clears OUT4 (blue LED)

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_2, SCT_EVT_1);													// event 1 sets OUT2 (red LED)
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_2, SCT_EVT_1); 											// event 1 clear OUT2 (red LED)

	Chip_SCT_SetConflictResolution(LPC_SCT, SCT_OUTPUT_4,SCT_RES_TOGGLE_OUTPUT);  // toggle OUT4 on conflict

	Chip_SCT_SetConflictResolution(LPC_SCT, SCT_OUTPUT_2,SCT_RES_TOGGLE_OUTPUT);	// toggle OUT2 on conflict

	Chip_SCT_Output(LPC_SCT, 0x10 );                   														// default set OUT4 and and clear OUT2

    Chip_SCT_ClearControl(LPC_SCT,SCT_CTRL_HALT_L);	   													// unhalt it by clearing bit 2 of the CTRL register
}


void SysTick_Handler(void)
{

	Chip_SCT_SetControl(LPC_SCT, SCT_CONFIG_NORELOADL_U);													// stop reload process for L counter

	if (Chip_GPIO_GetPinState(LPC_GPIO, 0, 24))              											// P0_24 high?
    {
        if (LPC_SCT->MATCHREL[2].L < hperiod-1)        													// check if DC2 < Period of PWM
        {
            LPC_SCT->MATCHREL[1].L ++;
            LPC_SCT->MATCHREL[2].L ++;
        }
    }
    else if (LPC_SCT->MATCHREL[1].L > 1)              													// check if DC1 > 1
    {
        LPC_SCT->MATCHREL[1].L --;
        LPC_SCT->MATCHREL[2].L --;
    }

    LPC_SCT->CONFIG &= ~(1 << 7);                     													// enable reload process for L counter
}

int main(void)
{
	SystemCoreClockUpdate();
	Board_Init();

	Chip_Clock_EnablePeriphClock((CHIP_SYSCON_CLOCK_T) ( 	SYSCON_CLOCK_GPIO0	|   // enable GPIO port 0 clock
															SYSCON_CLOCK_GPIO1	| 	        									// enable GPIO port 1 clock
															SYSCON_CLOCK_IOCON	|             								//enable IOCON clock
															SYSCON_CLOCK_INPUTMUX	));													//enable MUX clock


	Chip_GPIO_SetPinDIR(LPC_GPIO, 0, 24, false);

	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 24, 		IOCON_FUNC0 		|
												IOCON_I2C_SLEW		|
												IOCON_DIGITAL_EN	|
												IOCON_INPFILT_OFF	|
												0x0 << 6			|
												0x0 << 9			|
												0x1 << 10			);


	Chip_IOCON_PinMuxSet(LPC_IOCON, 1, 10, IOCON_FUNC3 | IOCON_MODE_INACT | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF);	//SCT0_OUT4 = P1.9  (green   LED)
	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 29, IOCON_FUNC2 | IOCON_MODE_INACT | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF);	//SCT0_OUT2 = P0.29  (red   LED)

	SysTick_Config(SystemCoreClock/50);                				// SysTick interrupt @ 50 Hz = 20 msec

	SCT_Init();                                          			// Initialize SCT0


    while (1)                                              	// loop forever
    {
    	__WFI();
    }
}
